// DAZ Store Hider — bg.js v4.0.0

// ---- Icon/Badge Management ----
function updateIcon(tab) {
  if (tab && tab.url && (tab.url.includes("daz3d.com") || tab.url.includes(chrome.runtime.getURL("popup.html")))) {
    chrome.action.setBadgeBackgroundColor({ color: '#00D100' });
  } else {
    chrome.action.setBadgeBackgroundColor({ color: '#8888' });
  }
  chrome.action.setBadgeText({ text: "DAZ" });
}

chrome.tabs.onActivated.addListener(activeInfo => {
  chrome.tabs.get(activeInfo.tabId, tab => {
    updateIcon(tab);
  });
});

chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (changeInfo.status === 'complete') {
    updateIcon(tab);
    // Inject state into content script on page load
    if (tab.url && tab.url.includes("daz3d.com")) {
      sendStateToTab(tabId);
    }
  }
});

// ---- Event Listeners ----
chrome.runtime.onInstalled.addListener(() => {
  chrome.tabs.query({ active: true, currentWindow: true }, tabs => {
    if (tabs[0]) updateIcon(tabs[0]);
  });
});

chrome.runtime.onStartup.addListener(() => {
  chrome.tabs.query({ active: true, currentWindow: true }, tabs => {
    if (tabs[0]) updateIcon(tabs[0]);
  });
});

// ---- Popup Singleton Management ----
let popupWindowId = null;
let popupTabId = null;

function openSingletonPopup(opts = {}, sendResponse) {
  const create = () => {
    chrome.windows.create(
      {
        url: chrome.runtime.getURL("popup.html"),
        type: "popup",
        width: opts.width ?? 816,
        height: opts.height ?? 640,
        top: opts.top ?? 50,
        left: opts.left ?? 100
      },
      (win) => {
        if (chrome.runtime.lastError || !win) {
          popupWindowId = null;
          popupTabId = null;
          sendResponse?.({ ok: false, error: chrome.runtime.lastError?.message });
          return;
        }

        popupWindowId = win.id ?? null;
        popupTabId = (win.tabs && win.tabs[0] && win.tabs[0].id) ? win.tabs[0].id : null;

        chrome.action.setBadgeBackgroundColor({ color: '#00D100' });
        chrome.action.setBadgeText({ text: "DAZ" });

        sendResponse?.({ ok: true, reused: false });
      }
    );
  };

  // If popup already exists, focus it
  if (popupWindowId !== null) {
    chrome.windows.get(popupWindowId, (win) => {
      if (chrome.runtime.lastError || !win) {
        popupWindowId = null;
        popupTabId = null;
        create();
        return;
      }

      chrome.windows.update(popupWindowId, { focused: true }, () => {
        if (popupTabId !== null) chrome.tabs.update(popupTabId, { active: true });
        sendResponse?.({ ok: true, reused: true });
      });
    });
    return true; // async
  }

  create();
  return true; // async
}

chrome.windows.onRemoved.addListener((windowId) => {
  if (windowId === popupWindowId) {
    popupWindowId = null;
    popupTabId = null;
  }
});

// If user clicks the extension action button, enforce singleton popup too
chrome.action.onClicked.addListener(() => {
  openSingletonPopup({ width: 1070, height: 664, top: 50, left: 100 });
});

// ---- Storage Keys ----
const N = 10;
const normArr = arr => Array.from({ length: N }, (_, i) => (arr && arr[i]) || { text: "", enabled: false });

const MASTER_KEY = "daz_hider_master_enabled";
const HIDE_KEY   = "daz_hider_rules";
const HL_KEY     = "daz_hider_highlight_rules";
const SHOW_KEY   = "daz_hider_show_rules";
const PRICE_KEY  = "daz_hider_price_rules";
const HIDE_COL   = "daz_hider_col_hide_enabled";
const HL_COL     = "daz_hider_col_hl_enabled";
const SHOW_COL   = "daz_hider_col_show_enabled";
const PRICE_COL  = "daz_hider_col_price_enabled";

// ---- Helper to send state to tab ----
async function sendStateToTab(tabId) {
  const keys = {
    [MASTER_KEY]: true,
    [HIDE_KEY]: [], [HL_KEY]: [], [SHOW_KEY]: [], [PRICE_KEY]: {},
    [HIDE_COL]: true, [HL_COL]: true, [SHOW_COL]: true, [PRICE_COL]: true
  };

  chrome.storage.sync.get(keys, (got) => {
    const state = {
      masterEnabled: got[MASTER_KEY] !== false,
      rules: normArr(got[HIDE_KEY]),
      highlightRules: normArr(got[HL_KEY]),
      showOnlyRules: normArr(got[SHOW_KEY]),
      priceRules: got[PRICE_KEY] || { min: "", max: "", minEnabled: false, maxEnabled: false },
      colHide: !!got[HIDE_COL],
      colHL: !!got[HL_COL],
      colShow: !!got[SHOW_COL],
      colPrice: !!got[PRICE_COL]
    };

    chrome.tabs.sendMessage(tabId, {
      type: "STATE_UPDATED",
      state: state
    }).catch(() => {
      // Content script might not be ready yet
    });
  });
}

// ---- Message Router ----
chrome.runtime.onMessage.addListener((m, sender, sendResponse) => {
  if (!m || !m.type) return;

  switch (m.type) {
    case "OPEN_POPUP": {
      // Enforce single instance
      return openSingletonPopup({ width: 816, height: 640, top: 50, left: 100 }, sendResponse);
    }

    case "GET_STATE": {
      const keys = {
        [MASTER_KEY]: true,
        [HIDE_KEY]: [], [HL_KEY]: [], [SHOW_KEY]: [], [PRICE_KEY]: {},
        [HIDE_COL]: true, [HL_COL]: true, [SHOW_COL]: true, [PRICE_COL]: true
      };

      chrome.storage.sync.get(keys, (got) => {
        sendResponse({
          state: {
            masterEnabled: got[MASTER_KEY] !== false,
            rules: normArr(got[HIDE_KEY]),
            highlightRules: normArr(got[HL_KEY]),
            showOnlyRules: normArr(got[SHOW_KEY]),
            priceRules: got[PRICE_KEY] || { min: "", max: "", minEnabled: false, maxEnabled: false },
            colHide: !!got[HIDE_COL],
            colHL: !!got[HL_COL],
            colShow: !!got[SHOW_COL],
            colPrice: !!got[PRICE_COL]
          }
        });
      });
      return true; // async
    }

    case "SET_STATE": {
      const p = m.payload;
      if (!p) { sendResponse({ ok: false }); return; }

      const newState = {
        [MASTER_KEY]: !!p.masterEnabled,
        [HIDE_KEY]: p.rules,
        [HL_KEY]: p.highlightRules,
        [SHOW_KEY]: p.showOnlyRules,
        [PRICE_KEY]: p.priceRules,
        [HIDE_COL]: !!p.colHide,
        [HL_COL]: !!p.colHL,
        [SHOW_COL]: !!p.colShow,
        [PRICE_COL]: !!p.colPrice
      };

      chrome.storage.sync.set(newState, () => {
        // Notify all DAZ tabs
        chrome.tabs.query({}, (tabs) => {
          tabs.forEach(tab => {
            if (tab.url && tab.url.includes("daz3d.com")) {
              sendStateToTab(tab.id);
            }
          });
        });

        chrome.tabs.query({ active: true, currentWindow: true }, tabs => {
          if (tabs[0]) updateIcon(tabs[0]);
        });

        sendResponse({ ok: !(chrome.runtime && chrome.runtime.lastError) });
      });

      return true; // async
    }

    case "FETCH_GALLERY": {
      const galleryUrl = "https://www.daz3d.com/gallery/user/5142885808734208#gallery=newest&page=1";
      
      fetch(galleryUrl)
        .then(response => response.text())
        .then(html => {
          // Use regex to parse the HTML since DOMParser is not available in service workers
          
          // Find the first gallery item
          const itemMatch = html.match(/<div[^>]*class="[^"]*gallery-item[^"]*"[^>]*>([\s\S]*?)<\/div>\s*<\/div>/);
          
          if (!itemMatch) {
            sendResponse({ success: false, error: "Could not find gallery item" });
            return;
          }
          
          const itemHtml = itemMatch[0];
          
          // Extract image src (look for data-src or src in img tag)
          const imgMatch = itemHtml.match(/<img[^>]*(?:data-src|src)="([^"]+)"[^>]*>/);
          const imageSrc = imgMatch ? imgMatch[1] : null;
          
          // Extract image URL (look for href in the link)
          const linkMatch = itemHtml.match(/<a[^>]*href="([^"]+)"[^>]*>/);
          const imageUrl = linkMatch ? "https://www.daz3d.com" + linkMatch[1] : null;
          
          // Extract title (look for title attribute or alt text)
          const titleMatch = itemHtml.match(/title="([^"]+)"|alt="([^"]+)"/);
          const imageTitle = titleMatch ? (titleMatch[1] || titleMatch[2]) : "Latest Image";
          
          if (imageSrc && imageUrl) {
            sendResponse({
              success: true,
              imageSrc: imageSrc,
              imageUrl: imageUrl,
              imageTitle: imageTitle
            });
          } else {
            sendResponse({ success: false, error: "Could not extract image data" });
          }
        })
        .catch(error => {
          sendResponse({ success: false, error: error.message });
        });
      
      return true; // async
    }
  }
});

// Gallery fetch handler
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === "FETCH_GALLERY") {
    fetch('https://www.daz3d.com/gallery/list/user/5142885808734208/newest?page=1', {
      credentials: 'omit'
    })
      .then(r => r.text())
      .then(html => {
        // Extract all page-ids (up to 10)
        const pageIdMatches = [...html.matchAll(/page-id="(\d+)"/g)];
        const imgSrcMatches = [...html.matchAll(/src="(data:image\/[^"]+)"/g)];
        
        if (pageIdMatches.length > 0 && imgSrcMatches.length > 0) {
          // Get up to 10 latest images
          const maxImages = Math.min(10, pageIdMatches.length, imgSrcMatches.length);
          const images = [];
          
          for (let i = 0; i < maxImages; i++) {
            const imageId = pageIdMatches[i][1];
            const imageSrc = imgSrcMatches[i][1];
            
            // Extract title for this specific image
            const titlePattern = new RegExp(`<h4><a href="#image=${imageId}"[^>]*>([^<]+)</a></h4>`, 'i');
            const titleMatch = html.match(titlePattern);
            const imageTitle = titleMatch ? titleMatch[1].trim() : `Image ${i + 1}`;
            
            images.push({
              imageSrc: imageSrc,
              imageUrl: `https://www.daz3d.com/gallery/#image=${imageId}`,
              imageTitle: imageTitle
            });
          }
          
          // Pick a random image from the collection
          const randomImage = images[Math.floor(Math.random() * images.length)];
          
          console.log('[DAZ Hider BG] Fetched', images.length, 'images, showing random one');
          console.log('[DAZ Hider BG] Selected:', randomImage.imageTitle);
          
          sendResponse({
            success: true,
            imageSrc: randomImage.imageSrc,
            imageUrl: randomImage.imageUrl,
            imageTitle: randomImage.imageTitle
          });
        } else {
          console.error('[DAZ Hider BG] Could not parse gallery HTML');
          sendResponse({ success: false, error: 'Could not parse gallery data' });
        }
      })
      .catch(error => {
        console.error('[DAZ Hider BG] Gallery fetch error:', error);
        sendResponse({ success: false, error: error.message });
      });
    
    return true;
  }
});